//	TorusGamesWindowController.m
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#import "TorusGamesWindowController.h"
#import "TorusGamesGraphicsViewMac.h"
#import "TorusGamesAppDelegate-Mac.h"	//	for -commandHelp:
#import "TorusGames-Common.h"
#import "GeometryGamesModel.h"
#import "GeometryGamesWindowMac.h"
#import "GeometryGamesUtilities-Mac-iOS.h"
#import "GeometryGamesColorSpaces.h"
#import "GeometryGamesLocalization.h"
#import "GeometryGamesBevelViewMac.h"


//	Don't let the window get too small.
//	Ideally the MIN_BOARD_SIZE should be
//	a multiple of 3 so the small fundamental domain
//		sits perfectly in the tiling view (NO LONGER SO IMPORTANT), and
//	a multiple of 4 to accommodate a 5:4 minimum aspect ratio.
#define MIN_BOARD_SIZE	240		//	points? pixels?

//	The game panel inset should be a multiple of three,
//	so the ViewBasicSmall image may exactly fill
//	the center 1/9 of the ViewRepeating image
//	(NO LONGER SO IMPORTANT).
//
//	Setting GP_CONTROL_PANEL_WIDTH_PT = 256 just barely
//	accommodates the two longest reset titles, namely
//
//		pt Tic-Tac-Toe	("Apagar o tabuleiro do jogo do galo")
//		el Word Search	("Αλλαγή πίνακας κρυμμένων λέξεων")
//
#define GP_CONTROL_PANEL_WIDTH_PT	256	//	in points, not pixels
#define GP_BOARD_INSET_PT			 30	//	in points, not pixels
#define GP_BEVEL_WIDTH_PT			  8	//	in points, not pixels
#define GP_BEVEL_INSET_PT			(GP_BOARD_INSET_PT - GP_BEVEL_WIDTH_PT)

//	Placement of control panel items
//	(Reset button, Change Game button and message view).
#define CP_BUTTON_HEIGHT			36		//	points? pixels?
#define CP_BUTTON_MARGIN_RIGHT		GP_BEVEL_INSET_PT
#define CP_BUTTON_MARGIN_V			 4
#define CP_MESSAGE_MARGIN_TOP		64
#define CP_HANZI_FIELD_HEIGHT		40

//	Layout of Change Game panel.
#define CGP_BUTTON_SIZE		 80
#define CGP_TEXT_HEIGHT		 32
#define CGP_V_SPACE_TOP		 24	//	above top row
#define CGP_V_SPACE_BOTTOM	  0	//	below bottom row
#define CGP_V_SPACE_INTRA	  4	//	between button and text label
#define CGP_V_STRIDE		(CGP_BUTTON_SIZE + CGP_V_SPACE_INTRA + CGP_TEXT_HEIGHT)
#define CGP_H_INSET			 18
#define CGP_H_STRIDE		(CGP_BUTTON_SIZE + 2*CGP_H_INSET)
#define CGP_H_MARGIN		 18

//	Parameters for spinning the board while the game resets.
#define RESET_ANIMATION_DURATION	1.0		//	seconds
#define RF1							0.25	//	reset shrinkage factor at 90° and 270°
#define RF2							0.0625	//	reset shrinkage factor at 180°


static NSString		*ChooseMessageFontName(void);
static double		ChooseMessageFontSize(double aViewWidth);


//	Privately-declared properties and methods
@interface TorusGamesWindowController()

- (void)userClickedResetGameButton:(id)sender;
- (NSString *)gameSpecificResetText;

- (void)showGameChoiceWindow:(id)sender;
- (void)userDidPushGameChoiceButton:(id)sender;
- (void)refreshGameChoiceLanguage;
- (void)refreshMessageCentering;

- (void)resetGameWithAnimationForPurpose:(ResetPurpose)aPurpose value:(unsigned int)aNewValue;
- (void)resetGameForPurpose:(ResetPurpose)aPurpose value:(unsigned int)aNewValue;

- (void)hanziTextDidChange:(NSNotification *)aNotification;
- (void)refreshHanziView;

@end


@implementation TorusGamesWindowController
{
	TorusGamesGraphicsViewMac	*itsGraphicsView;

	GeometryGamesBevelView		*itsBevelView;
	NSButton					*itsResetButton,
								*itsChangeGameButton;
	NSTextField					*itsMessageView,
								*itsHanziView;	//	for input to Chinese crossword puzzles
		
	//	Keep a pointer to the Change Game window while it's up,
	//	in case we need to change language.
	NSWindow					*itsGameChoiceWindow;
}


- (id)initWithDelegate:(id<GeometryGamesWindowControllerDelegate>)aDelegate
{
	self = [super initWithDelegate:aDelegate];
	if (self != nil)
	{
		//	No Game Choice window is present.
		itsGameChoiceWindow = nil;

		//	The superclass has created the window.
		//	Here we need only set its background color
		//	and minimum content size.
		[itsWindow setBackgroundColor:[NSColor
			//	colorWithDisplayP3Red:green:blue:alpha: takes gamma-encoded color components
			colorWithDisplayP3Red:	gMatteColorGammaP3[0]
			green:					gMatteColorGammaP3[1]
			blue:					gMatteColorGammaP3[2]
			alpha:					1.0]];
		[itsWindow setContentMinSize:(NSSize)
		{
			MIN_BOARD_SIZE + 2*GP_BOARD_INSET_PT + GP_CONTROL_PANEL_WIDTH_PT,
			MIN_BOARD_SIZE + 2*GP_BOARD_INSET_PT
		}];
		
#ifdef TORUS_GAMES_FOR_TALK
		[itsWindow toggleFullScreen:self];
#endif

#ifdef TORUS_GAMES_FOR_TALK
		[self changeGame:Game2DIntro];
#else
		//	Let the user select a game.
		//
		//	Delay the call to -showGameChoiceWindow, 
		//	so the user sees the sheet animate gracefully into place.
		[self performSelector:@selector(showGameChoiceWindow:) withObject:self afterDelay:0.125];
#endif
	}
	return self;
}


- (void)createSubviews
{
	//	Create the game view and the various buttons, etc.
	//	Don't worry about their placements just yet, 
	//	-windowDidResize: will place them momentarily.
	
	//	Create the bevel view
	itsBevelView = [[GeometryGamesBevelView alloc] initWithFrame:(NSRect){{0,0},{32,32}} bevelThickness:8
		red:gMatteColorGammaP3[0] green:gMatteColorGammaP3[1] blue:gMatteColorGammaP3[2]];
	[[itsWindow contentView] addSubview:itsBevelView];

	//	Create the game view

	itsGraphicsView = [[TorusGamesGraphicsViewMac alloc] initWithModel:itsModel frame:(NSRect){{8,8},{16,16}}];
	[itsGraphicsView setUpGraphics];
	[itsGraphicsView setUpDisplayLink];	//	but don't start it running until the view appears

	[[itsWindow contentView] addSubview:itsGraphicsView];

	//	Let the superclass treat itsGraphicsView as itsMainView.
	itsMainView = itsGraphicsView;

	//	Create the Reset button
	itsResetButton = [[NSButton alloc] initWithFrame:(NSRect){{0,0},{0,0}}];
	[itsResetButton setButtonType:NSButtonTypeMomentaryPushIn];
	[itsResetButton setTarget:self];
	[itsResetButton setAction:@selector(userClickedResetGameButton:)];
	[itsResetButton setTitle:[self gameSpecificResetText]];
	[itsResetButton setBordered:YES];
	[itsResetButton setBezelStyle:NSBezelStyleTexturedSquare];
	[[itsWindow contentView] addSubview:itsResetButton];

	//	Create the Change Game button
	itsChangeGameButton = [[NSButton alloc] initWithFrame:(NSRect){{0,0},{0,0}}];
	[itsChangeGameButton setButtonType:NSButtonTypeMomentaryPushIn];
	[itsChangeGameButton setTarget:self];
	[itsChangeGameButton setAction:@selector(showGameChoiceWindow:)];
	[itsChangeGameButton setTitle:GetLocalizedTextAsNSString(u"Change Game")];
	[itsChangeGameButton setBordered:YES];
	[itsChangeGameButton setBezelStyle:NSBezelStyleTexturedSquare];
	[[itsWindow contentView] addSubview:itsChangeGameButton];
	
	//	Create the message view
	itsMessageView = [[NSTextField alloc] initWithFrame:(NSRect){{0,0},{0,0}}];
	[itsMessageView setBezeled:NO];
	[itsMessageView setDrawsBackground:NO];
	[itsMessageView setSelectable:NO];
	[itsMessageView setAlignment:NSTextAlignmentCenter];
	[[itsWindow contentView] addSubview:itsMessageView];
	
	//	Create the hanzi view
	itsHanziView = [[NSTextField alloc] initWithFrame:(NSRect){{0,0},{0,0}}];
	[itsHanziView setBezeled:YES];
	[itsHanziView setDrawsBackground:YES];
	[itsHanziView setEditable:YES];
	[itsHanziView setSelectable:YES];
	[itsHanziView setBackgroundColor:[NSColor yellowColor]];
	[itsHanziView setAlignment:NSTextAlignmentCenter];
	[[itsWindow contentView] addSubview:itsHanziView];

	//	Ask to be notified when itsHanziView's text changes.
	[[NSNotificationCenter defaultCenter]
		addObserver:	self
		selector:		@selector(hanziTextDidChange:)
		name:			NSControlTextDidChangeNotification
		object:			itsHanziView];
}

- (void)handlePossibleGPUChangeNotification:(NSNotification *)aNotification
{
#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE
	//	Exit torus cursor mode, if necessary,
	//	before destroying itsGraphicsView.
	[itsGraphicsView exitTorusCursorMode];
#endif

	//	Re-create all subviews using the possibly newly-selected GPU.
	[super handlePossibleGPUChangeNotification:aNotification];
	
	//	Show or hide itsHanziView, as appropriate.
	[self refreshHanziView];
}

//	ModelData must not be locked, to avoid deadlock
//	when waiting for a possible CVDisplayLink callback to complete.
- (void)windowWillClose:(NSNotification *)aNotification
{
	//	Stop the CVDisplayLink and wait for it to finish drawing.
	[itsGraphicsView pauseAnimation];	//	ModelData must not be locked, to avoid deadlock
										//	when waiting for a possible CVDisplayLink callback to complete.

	//	Switch to the null game to free any allocated memory.
	[self changeGame:GameNone];

	//	Apple's Foundation Release Notes for macOS 10.11 and iOS 9 at
	//
	//		https://developer.apple.com/library/content/releasenotes/Foundation/RN-FoundationOlderNotes/index.html#10_11NotificationCenter
	//	says
	//		In OS X 10.11 and iOS 9.0 NSNotificationCenter and NSDistributedNotificationCenter
	//		will no longer send notifications to registered observers that may be deallocated.
	//		... This means that observers are not required to un-register in their deallocation method.
	//
//	[[NSNotificationCenter defaultCenter]
//		removeObserver:	self
//		name:			NSControlTextDidChangeNotification
//		object:			itsHanziView];
	
	//	The superclass will finish cleaning up
	//	and then release this controller.
	[super windowWillClose:aNotification];
}


- (NSRect)windowWillUseStandardFrame:(NSWindow *)window defaultFrame:(NSRect)newFrame
{
	NSRect	theContentRect;

	//	When the windows zooms, Cocoa calls this -windowWillUseStandardFrame:defaultFrame:.
	
	//	May 2017  THIS METHOD MAY ALREADY BE IRRELEVANT,
	//	NOW THAT "ZOOM" MEANS "ZOOM TO FULL SCREEN"
	
	//	Constrain the aspect ratio for windowed-mode zooms,
	//	but let fullscreen mode use the full screen.
	if ( ! ([itsWindow styleMask] & NSWindowStyleMaskFullScreen) )
	{
		//	Convert the newFrame to theContentRect.
		theContentRect = [window contentRectForFrameRect:newFrame];
		
		//	Don't drop below a 5:4 aspect ratio,
		//	so the control panel gets at least 1/5 of the view.
		//	Fullscreen mode on a 1280×1024 display has a 5:4 aspect ratio.
		//
		//	0.8 repeats in binary, so round slightly up.
		if (theContentRect.size.height > 0.80001 * theContentRect.size.width)
			theContentRect.size.height = 0.80001 * theContentRect.size.width;
		
		//	Don't go above a 4:3 aspect ratio,
		//	so the control panel gets at most 1/4 of the view.
		if (theContentRect.size.width  > 1.33334 * theContentRect.size.height)
			theContentRect.size.width  = 1.33334 * theContentRect.size.height;

		//	Stick to integers.
		theContentRect.size.width  = floor(theContentRect.size.width );
		theContentRect.size.height = floor(theContentRect.size.height);
		
		//	Convert theContentRect back to the newFrame.
		newFrame = [window frameRectForContentRect:theContentRect];
	}
	
	return newFrame;
}

- (void)windowDidResize:(NSNotification *)aNotification
{
	NSRect	theContentBounds;
	CGFloat	theGamePanelWidth,
			theGamePanelHeight,
			theMarginH,
			theMarginV;

	//	itsWindow's content view has been resized.
	//	Adjust the subviews accordingly.

	//	Note the available space.
	theContentBounds = [[itsWindow contentView] bounds];
	
	//	Start with tentative values for the game panel size.
	theGamePanelWidth	= theContentBounds.size.width - GP_CONTROL_PANEL_WIDTH_PT;
	theGamePanelHeight	= theContentBounds.size.height;
	
	//	Shrink the game panel as necessary to make it square.
	if (theGamePanelWidth  > theGamePanelHeight)
		theGamePanelWidth  = theGamePanelHeight;
	if (theGamePanelHeight > theGamePanelWidth )
		theGamePanelHeight = theGamePanelWidth;

	//	Note the margins.
	//	Use floor() to ensure alignment with the grid (in points).
	theMarginH = floor(0.5 * (theContentBounds.size.width - (theGamePanelWidth + GP_CONTROL_PANEL_WIDTH_PT)));
	theMarginV = floor(0.5 * (theContentBounds.size.height - theGamePanelHeight));

	[itsBevelView setFrame:(NSRect)
		{
			{
				theContentBounds.origin.x + theMarginH + GP_BEVEL_INSET_PT,
				theContentBounds.origin.y + theMarginV + GP_BEVEL_INSET_PT
			},
			{
				theGamePanelWidth  - 2*GP_BEVEL_INSET_PT,
				theGamePanelHeight - 2*GP_BEVEL_INSET_PT
			}
		}];

	[itsGraphicsView setFrame:(NSRect)
		{
			{
				theContentBounds.origin.x + theMarginH + GP_BOARD_INSET_PT,
				theContentBounds.origin.y + theMarginV + GP_BOARD_INSET_PT
			},
			{
				theGamePanelWidth  - 2*GP_BOARD_INSET_PT,
				theGamePanelHeight - 2*GP_BOARD_INSET_PT
			}
		}];

	[itsResetButton setFrame:(NSRect)
		{
			{
				theContentBounds.origin.x + theMarginH + theGamePanelWidth,
				theContentBounds.origin.y + theContentBounds.size.height
					- theMarginV
					- (GP_BEVEL_INSET_PT + CP_BUTTON_HEIGHT)
			},
			{
				GP_CONTROL_PANEL_WIDTH_PT - CP_BUTTON_MARGIN_RIGHT,
				CP_BUTTON_HEIGHT
			}
		}];

	[itsChangeGameButton setFrame:(NSRect)
		{
			{
				theContentBounds.origin.x + theMarginH + theGamePanelWidth,
				theContentBounds.origin.y + theContentBounds.size.height
					- theMarginV
					- (GP_BEVEL_INSET_PT + 2.0*CP_BUTTON_HEIGHT + CP_BUTTON_MARGIN_V)
			},
			{
				GP_CONTROL_PANEL_WIDTH_PT - CP_BUTTON_MARGIN_RIGHT,
				CP_BUTTON_HEIGHT
			}
		}];
	
	[itsMessageView setFrame:(NSRect)
		{
			{
				theContentBounds.origin.x + theMarginH + theGamePanelWidth,
				theContentBounds.origin.y + theMarginV + GP_BEVEL_INSET_PT + CP_HANZI_FIELD_HEIGHT
			},
			{
				GP_CONTROL_PANEL_WIDTH_PT - CP_BUTTON_MARGIN_RIGHT,
				theContentBounds.size.height
					- 2.0*theMarginV
					- GP_BEVEL_INSET_PT
					- 2.0*(CP_BUTTON_HEIGHT + CP_BUTTON_MARGIN_V)
					- CP_MESSAGE_MARGIN_TOP
					- CP_HANZI_FIELD_HEIGHT
			}
		}];
	[itsMessageView setFont:[NSFont
		fontWithName:	ChooseMessageFontName()
		size:			ChooseMessageFontSize([itsMessageView bounds].size.width)]];
	
	[itsHanziView setFrame:(NSRect)
		{
			{
				theContentBounds.origin.x + theMarginH + theGamePanelWidth,
				theContentBounds.origin.y + theMarginV + GP_BEVEL_INSET_PT
			},
			{
				GP_CONTROL_PANEL_WIDTH_PT - CP_BUTTON_MARGIN_RIGHT,
				CP_HANZI_FIELD_HEIGHT
			}
		}];

	//	Whether we use @"Heiti SC" or @"Heiti TC"
	//	makes no difference when the user is typing Pinyin,
	//	because itsHanziView shows only the Latin letters
	//	being used to compose the Hanzi.  As soon as
	//	the user selects a Hanzi character from among
	//	the text entry system's suggestions, that
	//	Hanzi immediately appears in the puzzle itself,
	//	and itsHanziView gets cleared.
	[itsHanziView setFont:[NSFont fontWithName:
		IsCurrentLanguage(u"zt") ? @"Heiti TC" : @"Heiti SC" size:24.0]];

	[[itsWindow contentView] setNeedsDisplay:YES];
}


- (BOOL)validateMenuItem:(NSMenuItem *)aMenuItem
{
	SEL				theAction;
	ModelData		*md				= NULL;
	GameType		theGame;
	bool			theHumanVsComputer;
	unsigned int	theDifficultyLevel;
	TopologyType	theTopology;
	bool			theShowGlideAxes;
	ViewType		theViewType;

	theAction = [aMenuItem action];

	[itsModel lockModelData:&md];
	theGame				= md->itsGame;
	theHumanVsComputer	= md->itsHumanVsComputer;
	theDifficultyLevel	= md->itsDifficultyLevel;
	theTopology			= md->itsTopology;
	theShowGlideAxes	= md->itsShowGlideAxes;
	theViewType			= md->itsViewType;
	[itsModel unlockModelData:&md];
	
	if (theAction == @selector(commandHumanVsComputer:))
	{
		if (theGame == Game2DTicTacToe
		 || theGame == Game2DGomoku
		 || theGame == Game2DChess
		 || theGame == Game2DPool
		 || theGame == Game3DTicTacToe)
		{
			[aMenuItem setState:([aMenuItem tag] == theHumanVsComputer ? NSControlStateValueOn : NSControlStateValueOff)];
			return YES;
		}
		else
		{
			[aMenuItem setState:NSControlStateValueOff];
			return NO;
		}
	}
	
	if (theAction == @selector(commandDifficultyLevel:))
	{
		if ((theGame == Game2DGomoku && theHumanVsComputer)
		 || theGame == Game2DMaze
		 || theGame == Game2DJigsaw
		 || (theGame == Game2DChess  && theHumanVsComputer)
		 || (theGame == Game2DPool   && theHumanVsComputer)
		 || theGame == Game2DApples
		 || theGame == Game3DMaze)
		{
			[aMenuItem setState:((unsigned int)[aMenuItem tag] == theDifficultyLevel ? NSControlStateValueOn : NSControlStateValueOff)];
			return YES;
		}
		else
		{
			[aMenuItem setState:NSControlStateValueOff];
			return NO;
		}
	}
	
	if (theAction == @selector(commandGlideReflections:))
	{
		if (theTopology == Topology2DKlein)
		{
			[aMenuItem setState:(theShowGlideAxes ? NSControlStateValueOn : NSControlStateValueOff)];
			return YES;
		}
		else
		{
			[aMenuItem setState:NSControlStateValueOff];
			return NO;
		}
	}

	if (theAction == @selector(commandTopology:))
	{
		[aMenuItem setHidden:(
			TopologyIs3D((TopologyType)[aMenuItem tag]) != GameIs3D(theGame))];
		[aMenuItem setState:([aMenuItem tag] == theTopology ? NSControlStateValueOn : NSControlStateValueOff)];
		return YES;
	}

	if (theAction == @selector(commandView:))
	{
		if (TopologyIs3D(theTopology))
		{
			switch ([aMenuItem tag])
			{
				case ViewBasicLarge:
					[aMenuItem setTitle:GetLocalizedTextAsNSString(u"Basic (3D)")];
					break;
				
				case ViewBasicSmall:
					[aMenuItem setTitle:LocalizationNotNeeded(@"n/a")];
					[aMenuItem setHidden:YES];
					break;
				
				case ViewRepeating:
					[aMenuItem setTitle:GetLocalizedTextAsNSString(u"Repeating (3D)")];
					break;
			}
		}
		else	//	2D 
		{
			switch ([aMenuItem tag])
			{
				case ViewBasicLarge:
					[aMenuItem setTitle:GetLocalizedTextAsNSString(u"Basic (2D)")];
					break;
				
				case ViewBasicSmall:
					[aMenuItem setTitle:GetLocalizedTextAsNSString(u"Basic - Small")];
					[aMenuItem setHidden:NO];
					break;
				
				case ViewRepeating:
					[aMenuItem setTitle:GetLocalizedTextAsNSString(u"Repeating (2D)")];
					break;
			}
		}
		[aMenuItem setState:([aMenuItem tag] == theViewType ? NSControlStateValueOn : NSControlStateValueOff)];
		return YES;
	}

	if (theAction == @selector(commandPracticeBoard:))
	{
		[aMenuItem setState:(theGame == Game2DIntro ? NSControlStateValueOn : NSControlStateValueOff)];
		return YES;
	}

	return [super validateMenuItem:aMenuItem];
}

- (void)commandHumanVsComputer:(id)sender
{
	ModelData	*md				= NULL;

	[itsModel lockModelData:&md];
	ChangeHumanVsComputer(md, (bool)[sender tag]);
	[itsModel unlockModelData:&md];
}

- (void)commandDifficultyLevel:(id)sender
{
	[self resetGameWithAnimationForPurpose:ResetWithNewDifficultyLevel value:(unsigned int)[sender tag]];
}

- (void)commandGlideReflections:(id)sender
{
	ModelData	*md				= NULL;

	[itsModel lockModelData:&md];
	SetShowGlideAxes(md, ! md->itsShowGlideAxes );
	[itsModel unlockModelData:&md];
}

//	ModelData must not be locked, to avoid deadlock
//	when waiting for a possible CVDisplayLink callback to complete.
- (void)commandSaveImageRGB:(id)sender
{
#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE
	[itsGraphicsView exitTorusCursorMode];
#endif

	[super commandSaveImageRGB:sender];		//	ModelData must not be locked, to avoid deadlock
											//	when waiting for a possible CVDisplayLink callback to complete.
}
- (void)commandSaveImageRGBA:(id)sender
{
#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE
	[itsGraphicsView exitTorusCursorMode];
#endif

	[super commandSaveImageRGBA:sender];	//	ModelData must not be locked, to avoid deadlock
											//	when waiting for a possible CVDisplayLink callback to complete.
}

- (void)commandTopology:(id)sender
{
	[self resetGameWithAnimationForPurpose:ResetWithNewTopology value:(TopologyType)[sender tag]];
}

- (void)commandView:(id)sender
{
	ModelData	*md				= NULL;

	[itsModel lockModelData:&md];
	ChangeViewType(md, (ViewType)[sender tag]);
	[itsModel unlockModelData:&md];
}

- (void)commandPracticeBoard:(id)sender
{
	ModelData	*md				= NULL;
	GameType	theGame;

	[itsModel lockModelData:&md];
	theGame = md->itsGame;
	[itsModel unlockModelData:&md];

	if (theGame != Game2DIntro)
		[self changeGame:Game2DIntro];
	else
		[self showGameChoiceWindow:self];
}


- (void)languageDidChange
{
	ModelData	*md				= NULL;

	[super languageDidChange];	//	refreshes window title

	[itsResetButton      setTitle:[self gameSpecificResetText]				];
	[itsChangeGameButton setTitle:GetLocalizedTextAsNSString(u"Change Game")	];

	[itsMessageView setFont:[NSFont
		fontWithName:	ChooseMessageFontName()
		size:			ChooseMessageFontSize([itsMessageView bounds].size.width)]];
	[self refreshMessageCentering];
	
	[self refreshHanziView];

	if (itsGameChoiceWindow != nil)
		[self refreshGameChoiceLanguage];

	//	Wait until the CVDisplayLink finishes it current redraw,
	//	so we can lock the ModelData.
	[itsModel lockModelData:&md];

	//	Refresh any game-specific text.
	switch (md->itsGame)
	{
		case Game2DCrossword:
		case Game2DWordSearch:
			//	Reset Crossword and Word Search to accommodate the new language.
			//	<game>Reset() will notice that the language has changed and 
			//	set the puzzle indices to 0, so the user will get the easiest
			//	puzzle in the new language.
			ResetGame(md);
			[itsGraphicsView refreshRendererTexturesForGameResetWithModelData:md];
			break;
		
		case Game2DPool:
		case Game2DApples:
		case Game3DMaze:
			//	No need to disturb the Pool game, Apples game or 3D Maze in progress.
			//	Just refresh the current status or instructions message in the new language.
			RefreshStatusMessageText(md);
			break;
		
		default:
			//	Do nothing.
			break;
	}

	//	Release our lock on the ModelData.
	[itsModel unlockModelData:&md];
}


- (void)userClickedResetGameButton:(id)sender
{
	[self resetGameWithAnimationForPurpose:ResetPlain value:0];
}

- (NSString *)gameSpecificResetText
{
	ModelData	*md				= NULL;
	Char16		*theResetKey;
	
	//	The Reset button's text depends on both the current game
	//	and the current language.

	[itsModel lockModelData:&md];
	switch (md->itsGame)
	{
		case GameNone:			theResetKey = u"Reset";						break;

		case Game2DIntro:		theResetKey = u"Reset Practice Board";		break;
		case Game2DTicTacToe:	theResetKey = u"Erase Tic-Tac-Toe Board";	break;
		case Game2DGomoku:		theResetKey = u"Clear Gomoku Board";		break;
		case Game2DMaze:		theResetKey = u"New Maze";					break;
		case Game2DCrossword:	theResetKey = u"New Crossword Puzzle";		break;
		case Game2DWordSearch:	theResetKey = u"New Word Search";			break;
		case Game2DJigsaw:		theResetKey = u"New Jigsaw Puzzle";			break;
		case Game2DChess:		theResetKey = u"Reset Chess Board";			break;
		case Game2DPool:		theResetKey = u"Reset Pool Balls";			break;
		case Game2DApples:		theResetKey = u"New Apples";				break;

		case Game3DTicTacToe:	theResetKey = u"Erase Tic-Tac-Toe Board";	break;
		case Game3DMaze:		theResetKey = u"New Maze";					break;

		default:				theResetKey = NULL;							break;
	}
	[itsModel unlockModelData:&md];

	return GetLocalizedTextAsNSString(theResetKey);
}


- (void)showGameChoiceWindow:(id)sender
{
	NSButton		*theButton;
	NSTextField		*theLabel;
	unsigned int	i,
					theRow,
					theCol;
	
	static const struct
	{
		GameType	itsGameType;
		Char16		*itsTitleKey,
					*itsImageName;
		
	} theButtonData[] =
	{
		//	Omit the .png suffix, otherwise the image loader won't find the file.
		{Game2DTicTacToe,	u"Tic-Tac-Toe",		u"2D TicTacToe"		},
#ifdef TORUS_GAMES_FOR_TALK
		//	Suppress Gomoku during talks -- it's not so familiar to Western audiences,
		//	and in any case it conveys the topology less immediately.
#else
		{Game2DGomoku,		u"Gomoku",			u"2D Gomoku"		},
#endif
		{Game2DMaze,		u"Maze",			u"2D Maze"			},
		{Game2DCrossword,	u"Crossword",		u"2D Crossword"		},
		{Game2DWordSearch,	u"Word Search",		u"2D Word Search"	},
		{Game2DJigsaw,		u"Jigsaw Puzzle",	u"2D Jigsaw"		},
		{Game2DChess,		u"Chess",			u"2D Chess"			},
		{Game2DPool,		u"Pool",			u"2D Pool"			},
#ifdef TORUS_GAMES_FOR_TALK
		//	Suppress Apples during talks -- it's a more local game,
		//	with less dependence on the topology.
#else
		{Game2DApples,		u"Apples",			u"2D Apples"		},
#endif
		{Game3DTicTacToe,	u"3D Tic-Tac-Toe",	u"3D TicTacToe"		},
		{Game3DMaze,		u"3D Maze",			u"3D Maze"			},
#ifdef TORUS_GAMES_FOR_TALK
		{Game2DIntro,		u"Practice Board",	u"2D Practice Board"}
#else
		{Game2DIntro,		u"Help",			u"Question Mark"	}
#endif
	};
	
	static const unsigned int	theNumButtons	= BUFFER_LENGTH(theButtonData),
#ifdef TORUS_GAMES_FOR_TALK
								theNumCols		= 4,
#else
								theNumCols		= 3,
#endif
								theNumRows		= (theNumButtons + (theNumCols - 1)) / theNumCols;
		
	//	Create itsGameChoiceWindow.
	itsGameChoiceWindow = [[NSWindow alloc]
		initWithContentRect:	(NSRect)
							{
								{
									0,
									0
								}, 
								{
									theNumCols*CGP_H_STRIDE + 2*CGP_H_MARGIN,
									theNumRows*CGP_V_STRIDE + CGP_V_SPACE_TOP + CGP_V_SPACE_BOTTOM
								}
							}
		styleMask:				NSWindowStyleMaskTitled
		backing:				NSBackingStoreBuffered
		defer:					NO];

	//	Add the buttons and text labels.
	for (i = 0; i < theNumButtons; i++)
	{
		theRow = i / theNumCols;
		theCol = i % theNumCols;
		
		if (theNumCols == 3 && theNumButtons == 10 && i == 9)
			theCol = 1;	//	ad hoc code to center the Help button
	
		theButton = [[NSButton alloc] initWithFrame:(NSRect)
					{
						{
							CGP_H_MARGIN + CGP_H_INSET + theCol*CGP_H_STRIDE,
							CGP_V_SPACE_BOTTOM + theNumRows*CGP_V_STRIDE - CGP_BUTTON_SIZE - theRow*CGP_V_STRIDE
						},
						{
							CGP_BUTTON_SIZE,
							CGP_BUTTON_SIZE
						}
					}];
		[theButton setButtonType:NSButtonTypeMomentaryPushIn];
		[theButton setTag:theButtonData[i].itsGameType];
		[theButton setTarget:self];
		[theButton setAction:@selector(userDidPushGameChoiceButton:)];
		[theButton setImage:[NSImage imageNamed:
			//	Note:  The images themselves are in Display P3,
			//	but somehow that information is getting lost.
			//	Given that the legacy macOS version of Torus Games
			//	will soon be superseded by the iOS version
			//	running on Apple Silicon Macs, I'm not going to waste
			//	any more time trying to fix these.  The buttons show up
			//	with less saturated colors than intended, but that's
			//	no disaster.
			[@"Game Choice Buttons/" stringByAppendingString:
				GetNSStringFromZeroTerminatedString(theButtonData[i].itsImageName)]]];
		[theButton setBordered:YES];
		[theButton setBezelStyle:NSBezelStyleTexturedSquare];
		[[itsGameChoiceWindow contentView] addSubview:theButton];
		
		theLabel = [[NSTextField alloc] initWithFrame:(NSRect)
					{
						//	Widen theLabel from width CGP_H_STRIDE
						//	to width CGP_H_STRIDE + 2*CGP_H_MARGIN, to allow 
						//	for long game names like "Boter, kaas en eieren" in Dutch
						//	and「クロスワード・パズル」in Japanese.
						//	Neighboring label fields will overlap, 
						//	but in practice no collisions occur.
						{
							0 + theCol*CGP_H_STRIDE,
							(theNumRows*CGP_V_STRIDE + CGP_V_SPACE_TOP + CGP_V_SPACE_BOTTOM) - (CGP_V_STRIDE + CGP_V_SPACE_TOP) - theRow*CGP_V_STRIDE
						},
						{
							CGP_H_STRIDE + 2*CGP_H_MARGIN,
							CGP_TEXT_HEIGHT
						}
					}];
		[theLabel setStringValue:GetLocalizedTextAsNSString(theButtonData[i].itsTitleKey)];
		[theLabel setTag:(NSInteger)theButtonData[i].itsTitleKey];	//	useful when changing language on the fly
		[theLabel setBezeled:NO];
		[theLabel setDrawsBackground:NO];
		[theLabel setSelectable:NO];
		[theLabel setAlignment:NSTextAlignmentCenter];
		[[itsGameChoiceWindow contentView] addSubview:theLabel];
	}

	//	itsGameChoiceWindow shouldn't prevent the user from quitting.
	[itsGameChoiceWindow setPreventsApplicationTerminationWhenModal:NO];

	//	Display itsGameChoiceWindow as a sheet.
	//	-beginSheet:completionHandler: returns immediately.
	//	When the user clicks a button, it will call -userDidPushGameChoiceButton
	//	and we'll dismiss itsGameChoiceWindow from there.
	[itsWindow beginSheet:itsGameChoiceWindow completionHandler:^(NSModalResponse returnCode){}];
}

- (void)userDidPushGameChoiceButton:(id)sender
{
	NSButton	*theButton;
	GameType	theNewGame;

	if ([sender isKindOfClass:[NSButton class]])
	{
		theButton	= (NSButton *) sender;
		theNewGame	= (GameType) [theButton tag];
		
		[itsWindow endSheet:itsGameChoiceWindow];
		itsGameChoiceWindow	= nil;

		[self changeGame:theNewGame];

#ifdef TORUS_GAMES_FOR_TALK
#else
		//	If the user asked for Game2DIntro,
		//	show the Wraparound Universe page.
		if (theNewGame == Game2DIntro)
			[(TorusGamesAppDelegate *)[NSApp delegate] commandHelp:nil];
#endif
	}
}

- (void)refreshGameChoiceLanguage
{
	NSView				*theContentView;
	NSArray<NSView *>	*theSubviews;
	NSView				*theSubview;
	NSTextField			*theLabel;
	Char16				*theKey;

	if (itsGameChoiceWindow != nil)
	{
		theContentView	= [itsGameChoiceWindow contentView];
		theSubviews		= [[theContentView subviews] copy];	//	NSView documentation says to copy before using.

		//	itsGameChoiceWindow contains buttons and labels.
		//	Each label displays the localized name for the game 
		//	(for example "Tres en Raya", "三目並べ" or "Крестики-нолики"),
		//	but also its tag keeps a pointer to the underlying key ("Tic-Tac-Toe")
		//	so that we may re-localize the displayed name on the fly.
		for (theSubview in theSubviews)
		{
			if ([theSubview isKindOfClass:[NSTextField class]])
			{
				theLabel	= (NSTextField *)theSubview;
				theKey		= (Char16 *)[theLabel tag];

				[theLabel setStringValue:GetLocalizedTextAsNSString(theKey)];
			}
		}
	}
}

- (void)changeGame:(GameType)aNewGame
{
	[self resetGameWithAnimationForPurpose:ResetWithNewGame value:aNewGame];
}

- (void)refreshMessageCentering
{
	ModelData	*md				= NULL;
	GameType	theGame;

	[itsModel lockModelData:&md];
	theGame = md->itsGame;
	[itsModel unlockModelData:&md];

	//	The Japanese Word Search list looks best left aligned,
	//	so that the characters align vertically.
	//	For all other games and languages, center aligned text looks best.
	if (theGame == Game2DWordSearch && IsCurrentLanguage(u"ja"))
		[itsMessageView setAlignment:NSTextAlignmentLeft];
	else
		[itsMessageView setAlignment:NSTextAlignmentCenter];
}

- (void)setTorusGamesStatusMessage:(NSString *)aText color:(ColorP3Linear)aColorP3Linear
{
	[itsMessageView setStringValue:aText];

	[itsMessageView setTextColor:
		//	colorWithDisplayP3Red:green:blue:alpha: takes gamma-encoded color components
		[NSColor colorWithDisplayP3Red:	GammaEncode(aColorP3Linear.r)
								 green:	GammaEncode(aColorP3Linear.g)
								  blue:	GammaEncode(aColorP3Linear.b)
								 alpha:	aColorP3Linear.a]];
}


#pragma mark -
#pragma mark reset animation

- (void)resetGameWithAnimationForPurpose:(ResetPurpose)aPurpose value:(unsigned int)aNewValue
	//	aNewValue may be a new GameType, new TopologyType or new difficulty level,
	//	according to aPurpose
{
	ModelData	*md				= NULL;
	GameType	theGame;
	
	//	We've got three ways to reset a game:
	//
	//		-resetGameForPurpose:value:
	//			resets the game immediately, with no animation.
	//
	//		Reset3DGameWithAnimation()
	//			resets a 3D game with a spinning cube (in ViewBasicLarge)
	//			or fade in/out animation (in ViewRepeating),
	//			using Metal directly (no Core Animation).
	//
	//		-resetGameWithSpinningSquareAnimationForPurpose:value:
	//			resets the game with a spinning square animation,
	//			applied to itsGameEnclosureView, using Core Animation.
	//			[I ORIGINALLY AVOIDED THIS THIRD APPROACH ON macOS
	//			BECAUSE THERE WERE PROBLEMS USING A CUSTOM OPENGL SUBVIEW
	//			IN A LAYER-BACKED VIEW.]
	//
	//	The present function makes the purely aesthetic decision
	//	of which animation to apply in which circumstances.
	//	This decision may be revised at the programmer's whim,
	//	subject only to the constraint that the second method listed above
	//	applies only to 3D games, not 2D ones.

	//	Note the current game.
	[itsModel lockModelData:&md];
	theGame = md->itsGame;
	[itsModel unlockModelData:&md];
	
	//	If the old game and/or the new game is GameNone,
	//	don't animate the transition.
	if (theGame == GameNone
	 || (aPurpose == ResetWithNewGame && aNewValue == GameNone))
	{
		[self resetGameForPurpose:aPurpose value:aNewValue];
	}
	else
	//	If we're not changing games, and the game is 3D,
	//	then animate the transition using Metal directly.
	if (aPurpose != ResetWithNewGame &&	GameIs3D(theGame))
	{
		[itsModel lockModelData:&md];
		Reset3DGameWithAnimation(md, aPurpose, aNewValue);
		[itsModel unlockModelData:&md];
	}
	else
	//	In all remaining cases, animate the transition using Core Animation.
	{
		//	The spinning square doesn't work on macOS.  See comments above.
		//
		//		[self resetGameWithSpinningSquareAnimationForPurpose:aPurpose value:aNewValue];
		//
		//	Instead, do an immediate reset, with no animation.
		[self resetGameForPurpose:aPurpose value:aNewValue];
	}
}

- (void)resetGameForPurpose:(ResetPurpose)aPurpose value:(unsigned int)aNewValue
{
	GameType		theNewGame;
	TopologyType	theNewTopology;
	unsigned int	theNewDifficultyLevel;
	ModelData		*md				= NULL;
	
	//	The iOS Torus Games use a slightly fancier version
	//	of this function, which gets called as part
	//	of the spinning square animation.

	[itsModel lockModelData:&md];
	
	switch (aPurpose)
	{
		case ResetPlain:
			ResetGame(md);
			[itsGraphicsView refreshRendererTexturesForGameResetWithModelData:md];
			break;
		
		case ResetWithNewGame:
		
			//	Let's handle ResetWithNewGame here in the platform-dependent code,
			//	rather than in a platform-independent ChangeGame() function,
			//	so we can update the platform-dependent UI with no need for callbacks.
		
			//	Note the new game.
			theNewGame = (GameType)aNewValue;

			//	If the user re-selects the game that's already running,
			//	leave that game untouched.  Don't reset it.
			if (theNewGame != md->itsGame)
			{
				//	Shut down the old game.
				ShutDownGame(md);
		
				//	Switch to the new game.
				SetNewGame(md, theNewGame);
			
				//	Clear the message text.
				[itsMessageView setStringValue:@""];
				
				//	Set up the new game.
				SetUpGame(md);

				//	Request new textures.
				//
				//	Request updates to Metal textures before unlocking the ModelData,
				//	to ensure that the rendering code finds the ModelData and the Metal textures
				//	in a consistent state.
				//
				[itsGraphicsView refreshRendererTexturesWithModelData:md];
			}
			
			//	Wait to call -refreshHanziView etc until after we've unlocked the ModelData.
	
			break;
		
		case ResetWithNewTopology:

			theNewTopology = (TopologyType)aNewValue;

			//	ChangeTopology() always resets the game.
			ChangeTopology(md, theNewTopology);
			
			[itsGraphicsView refreshRendererTexturesWithModelData:md];

			break;
		
		case ResetWithNewDifficultyLevel:

			theNewDifficultyLevel = aNewValue;

			ChangeDifficultyLevel(md, theNewDifficultyLevel);

			[itsGraphicsView refreshRendererTexturesWithModelData:md];
			
			break;
	}

	[itsModel unlockModelData:&md];

	//	Show or hide itsHanziView as appropriate.
	[self refreshHanziView];

	if (aPurpose == ResetWithNewGame)
	{
		//	Update the Reset button to reflect the new game.
		//	(Do this after unlocking the ModelData,
		//	because -gameSpecificResetText will want to lock it.)
		[itsResetButton setTitle:[self gameSpecificResetText]];

		//	Refresh itsMessageView's centering.
		[self refreshMessageCentering];
	}
}


#pragma mark -
#pragma mark hanzi

- (void)hanziTextDidChange:(NSNotification *)aNotification
{
	NSString	*theHanziString;
	
	if ([aNotification object] == itsHanziView)	//	should never fail
	{
		theHanziString = [itsHanziView stringValue];
		if ([theHanziString length] > 0)
		{
			[itsGraphicsView insertText:theHanziString];
			[itsHanziView setStringValue:@""];
		}
	}
}

- (void)refreshHanziView
{
	ModelData	*md				= NULL;
	GameType	theGame;

	//	Clear any pending text.
	//	[NOPE.  THE PROBLEM IS:
	//
	//		Calling
	//
	//			[itsHanziView setStringValue:@""]
	//
	//		here triggers a call to -hanziTextDidChange,
	//		whose call [itsHanziView stringValue] returns
	//		the itsHanziView's old value (!), including
	//		any pending characters that might have been present
	//		(for example, Latin pinyin letters waiting
	//		to be assembled into hanzi).  So better not
	//		to clear itsHanziView.
	//	]
//	[itsHanziView setStringValue:@""];
	
	//	Show itsHanziView only for the Crossword and only in Chinese.
	[itsModel lockModelData:&md];
	theGame = md->itsGame;
	[itsModel unlockModelData:&md];
	if (theGame == Game2DCrossword
	 && (IsCurrentLanguage(u"zs") || IsCurrentLanguage(u"zt")))
	{
		//	Show itsHanziView.
		[itsHanziView setHidden:NO];
		
		//	Make itsHanziView the first responder,
		//	so it will get keystrokes.  Unfortunately
		//	this means that it will also get NSMouseMoved events,
		//	which it will send up the responder chain.
		[itsWindow makeFirstResponder:itsHanziView];
		
		//	For reasons I don't understand,
		//
		//		when an action message (for example, a menu command)
		//			travels up the responder chain and reaches the window,
		//			the window offers it to its delegate,
		//	but
		//		when an NSMouseMoved event
		//			travels up the responder chain and reaches the window,
		//			the window does *not* offer it to its delegate.
		//
		//	In order to receive -mouseMoved messages (see code below)
		//	in this window controller, we must explicitly insert it
		//	into the responder chain.
		if ([itsWindow nextResponder] != self)
		{
			[self setNextResponder:[itsWindow nextResponder]];	//	is typically nil
			[itsWindow setNextResponder:self];
		}
	}
	else
	{
		//	Hide itsHanziView.
		[itsHanziView setHidden:YES];
		
		//	Make itsGraphicsView the first responder, so it will
		//	receive keystrokes and NSMouseMoved events directly.
		[itsWindow makeFirstResponder:itsGraphicsView];
		
		//	It's harmless to leave this window controller in the responder chain.
	}
}

- (void)mouseMoved:(NSEvent *)anEvent
{
	//	If this function gets called, it means
	//	that the Crossword puzzle is running in Chinese.
	//	We made itsHanziView the first responder
	//	so it would receive keystrokes.
	//	As an undesirable side effect, it also receives
	//	NSMouseMoved events, which it sends up the responder chain.
	//	Forward such NSMouseMoved events to itsGraphicsView,
	//	which is where we would have liked them to have gone
	//	to begin with.
	[itsGraphicsView mouseMoved:anEvent];
}


@end


#pragma mark -
#pragma mark message font

static NSString *ChooseMessageFontName(void)
{
	//	If desired, the Crossword clue, Word Search word list
	//	and Pool status message could each use a different font face.
	//	At the moment, though, they all use the same font face,
	//	so we might as well keep the code that selects it in one place.
	
	if (IsCurrentLanguage(u"ja"))
	{
		//	"Hiragino Kaku Gothic Pro W3" and "Hiragino Kaku Gothic Pro W6"
		//		are gothic fonts of lighter and heavier line weights.
		//	"Hiragino Mincho Pro W3" and "Hiragino Mincho Pro W6"
		//		are mincho fonts of lighter and heavier line weights.
		//	"Arial Unicode MS" is a gothic font with a fairly heavy line weight.
		return @"Hiragino Kaku Gothic Pro W3";
	}
	else
	if (IsCurrentLanguage(u"ko"))
	{
		//	"AppleMyungjo Regular" is easy to read,
		//		yet still has a human touch.
		//	"AppleGothic Regular" is very easy to read,
		//		but lacks AppleMyungjo's human touch.
		//	"GungSeo Regular" (궁서) has personality,
		//		but the characters seems a little small,
		//		while the spacing between them seems a little large.
		//	"PilGi Regular" looks like handwriting.
		//		OK, but maybe a little hard to read.
		//	"Arial Unicode MS" is like "AppleGothic Regular"
		//		but slightly bolder.
		//
		//	Note that macOS is fussy about font names:
		//	"GungSeo Regular" works while "GungSeo" fails, yet
		//	"Arial Unicode MS Regular" fails while "Arial Unicode MS" works.
		//	Go figure.
		//
		return @"AppleGothic Regular";
	}
	else
	if (IsCurrentLanguage(u"zs"))
	{
		//	For options, the page
		//		http://www.yale.edu/chinesemac/pages/fonts.html
		//	contains a chart "Fonts via Apple" that says
		//	which fonts are included in which versions of OS.
		return @"Heiti SC";
	}
	else
	if (IsCurrentLanguage(u"zt"))
	{
		//	For options, see comment immediately above.
		return @"Heiti TC";
	}
	else
		return @"Helvetica";
}

static double ChooseMessageFontSize(
	double	aViewWidth)
{
	double	theFullFontSize,
			theFullViewWidth;
	
	//	Japanese needs a slightly smaller font size
	//	to allow room for the pool status messages,
	//	which have been pre-split using hard newlines.

	if (IsCurrentLanguage(u"ja"))
	{
		theFullFontSize		=  24.0;
		theFullViewWidth	= 272.0;
	}
	else
	{
		theFullFontSize		=  24.0;
		theFullViewWidth	= 192.0;
	}
	
	if (aViewWidth >= theFullViewWidth)
		return theFullFontSize;
	else
		return theFullFontSize * (aViewWidth / theFullViewWidth);
}
